<?php
declare(strict_types=1);
require_once __DIR__ . '/helpers.php';

function db(): PDO {
  static $pdo = null;
  if ($pdo) return $pdo;

  $needInit = !file_exists(DB_PATH);

  // Ensure data dir exists
  $dataDir = dirname(DB_PATH);
  if (!is_dir($dataDir)) mkdir($dataDir, 0775, true);

  $pdo = new PDO('sqlite:' . DB_PATH, null, null, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
  $pdo->exec('PRAGMA foreign_keys = ON;');

  if ($needInit) {
    migrate($pdo);
    seed_admin($pdo);
  } else {
    // Ensure schema exists (light check)
    $pdo->exec('PRAGMA foreign_keys = ON;');
  }

  return $pdo;
}

function migrate(PDO $pdo): void {
  $schema = <<<SQL
PRAGMA foreign_keys = ON;

CREATE TABLE IF NOT EXISTS users (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  role TEXT NOT NULL CHECK(role IN ('user','owner','admin')),
  name TEXT NOT NULL,
  email TEXT NOT NULL UNIQUE,
  phone TEXT,
  password_hash TEXT NOT NULL,
  created_at TEXT NOT NULL DEFAULT (datetime('now'))
);

CREATE TABLE IF NOT EXISTS cars (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  owner_id INTEGER NOT NULL,
  status TEXT NOT NULL DEFAULT 'pending' CHECK(status IN ('pending','active','suspended')),
  title TEXT NOT NULL,
  make TEXT,
  model TEXT,
  year INTEGER,
  transmission TEXT,
  seats INTEGER,
  city TEXT,
  pickup_address TEXT,
  price_per_day INTEGER NOT NULL,
  deposit INTEGER NOT NULL DEFAULT 0,
  booking_mode TEXT NOT NULL DEFAULT 'instant' CHECK(booking_mode IN ('instant','request')),
  rules TEXT,
  created_at TEXT NOT NULL DEFAULT (datetime('now')),
  FOREIGN KEY(owner_id) REFERENCES users(id)
);

CREATE TABLE IF NOT EXISTS car_photos (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  car_id INTEGER NOT NULL,
  path TEXT NOT NULL,
  sort_order INTEGER NOT NULL DEFAULT 0,
  FOREIGN KEY(car_id) REFERENCES cars(id) ON DELETE CASCADE
);

CREATE TABLE IF NOT EXISTS car_blocks (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  car_id INTEGER NOT NULL,
  start_date TEXT NOT NULL,
  end_date TEXT NOT NULL,
  reason TEXT,
  FOREIGN KEY(car_id) REFERENCES cars(id) ON DELETE CASCADE
);

CREATE TABLE IF NOT EXISTS bookings (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  car_id INTEGER NOT NULL,
  renter_id INTEGER NOT NULL,
  start_date TEXT NOT NULL,
  end_date TEXT NOT NULL,
  days INTEGER NOT NULL,
  subtotal INTEGER NOT NULL,
  service_fee INTEGER NOT NULL,
  total INTEGER NOT NULL,
  status TEXT NOT NULL DEFAULT 'requested'
    CHECK(status IN ('requested','approved','rejected','pending_payment','paid','cancelled')),
  created_at TEXT NOT NULL DEFAULT (datetime('now')),
  FOREIGN KEY(car_id) REFERENCES cars(id),
  FOREIGN KEY(renter_id) REFERENCES users(id)
);

CREATE TABLE IF NOT EXISTS payments (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  booking_id INTEGER NOT NULL,
  method TEXT NOT NULL CHECK(method IN ('bank_transfer','dlocal')),
  status TEXT NOT NULL DEFAULT 'pending' CHECK(status IN ('pending','confirmed','failed')),
  amount INTEGER NOT NULL,
  reference TEXT,
  created_at TEXT NOT NULL DEFAULT (datetime('now')),
  FOREIGN KEY(booking_id) REFERENCES bookings(id) ON DELETE CASCADE
);

CREATE INDEX IF NOT EXISTS idx_cars_city ON cars(city);
CREATE INDEX IF NOT EXISTS idx_bookings_car_dates ON bookings(car_id, start_date, end_date);
SQL;
  $pdo->exec($schema);
}

function seed_admin(PDO $pdo): void {
  $email = 'admin@movecarsharing.com';
  $exists = $pdo->prepare('SELECT id FROM users WHERE email = ?');
  $exists->execute([$email]);
  if ($exists->fetch()) return;

  $hash = password_hash('Admin123!', PASSWORD_DEFAULT);
  $stmt = $pdo->prepare('INSERT INTO users (role, name, email, phone, password_hash) VALUES (?,?,?,?,?)');
  $stmt->execute(['admin', 'Admin', $email, '', $hash]);
}
